package ndbclient

import (
	"context"
	"fmt"
	"net/http"

	gojson "github.com/gogf/gf/v2/encoding/gjson"
	"github.com/gogf/gf/v2/frame/g"
	"github.com/tidwall/gjson"
)

var (
	client  = g.Client()
	gOption Option
)

type Option struct {
	User string
	Pwd  string
	Addr string
	// MaxIdleConnsPerHost 设置每个host最大空闲连接数，通常5个，需要根据业务情况调整
	MaxIdleConnsPerHost int
}

func Init(option Option) {
	gOption = option
	// 设置连接池
	if gOption.MaxIdleConnsPerHost == 0 {
		gOption.MaxIdleConnsPerHost = 5
	}
	client.Transport.(*http.Transport).MaxIdleConnsPerHost = gOption.MaxIdleConnsPerHost
}

func request(ctx context.Context, urlPath string, arg map[string]interface{}) Response {
	body := map[string]interface{}{
		"arg": arg,
		"auth": map[string]interface{}{
			"user": gOption.User,
			"pwd":  gOption.Pwd,
		},
	}
	finalUrl := gOption.Addr + urlPath
	r, err := client.ContentJson().Post(ctx, finalUrl, body)
	if err != nil {
		return newResponse(ctx, finalUrl, body, err, nil)
	}
	defer r.Close()
	return newResponse(ctx, finalUrl, body, err, r.ReadAll())
}

type Response struct {
	response []byte
	result   gjson.Result
	url      string
	body     map[string]interface{}
	err      error
	errMsg   string
	ctx      context.Context
}

func newResponse(ctx context.Context, url string, body map[string]interface{}, err error, response []byte) Response {
	r := Response{
		ctx:      ctx,
		url:      url,
		body:     body,
		err:      err,
		response: response,
		result:   gjson.ParseBytes(response),
	}
	r.checkErr()
	// 全局输出错误日志
	if !r.IsSuccess() {
		g.Log().Errorf(r.ctx, "%s", r.err.Error())
	}
	return r
}

func (r *Response) checkErr() {
	jsonBody, err := gojson.New(r.body).ToJsonString()
	if err != nil {
		r.err = fmt.Errorf("request url: %s,body: %v, body json_encode err: %w", r.url, r.body, err)
		r.errMsg = "arg is invalid json"
		return
	}

	var _response string
	if r.response != nil {
		if len(_response) > 10000 {
			_response = string(r.response[:10000]) + "..."
		} else {
			_response = string(r.response)
		}
	}
	if r.err != nil {
		r.err = fmt.Errorf("request url: %s, body: %s, response:%s, error: %w", r.url, jsonBody, _response, r.err)
		r.errMsg = r.err.Error()
		return
	}
	if !r.result.Get("code").Exists() {
		r.err = fmt.Errorf("request url: %s, body: %s, code: %s, response: %s", r.url, jsonBody, "null", _response)
		r.errMsg = fmt.Sprintf("code: %s, response: %s", "null", _response)
		return
	}
	if r.result.Get("code").Int() != 0 {
		r.err = fmt.Errorf("request url: %s, body: %s, code: %d, response: %s, error: %s", r.url, jsonBody, r.result.Get("code").Int(), _response, r.result.Get("msg").String())
		r.errMsg = fmt.Sprintf("code: %d, error: %s", r.result.Get("code").Int(), r.result.Get("msg").String())
		return
	}
	r.err = nil
	r.errMsg = ""
}

func (r *Response) IsSuccess() bool {
	return r.err == nil
}

func (r *Response) GetErrMsg() string {
	return r.errMsg
}

func (r *Response) GetResult() gjson.Result {
	return r.result
}
