package ndbclient

import (
	"fmt"
	"strings"
)

// DuckDBSqlBuilder DuckDB SQL 构建器实现
type DuckDBSqlBuilder struct {
	fields    []string
	formId    string
	where     []Where
	whereArgs []interface{}
	limitPage int
	limitSize int
	groupBy   string
	orderBy   string
}

// NewDuckDBSqlBuilder 创建新的 DuckDB SQL 构建器
func NewDuckDBSqlBuilder() SqlBuilder {
	return &DuckDBSqlBuilder{
		fields:    make([]string, 0),
		where:     make([]Where, 0),
		whereArgs: make([]interface{}, 0),
	}
}

// Select 实现 Select 方法
func (d *DuckDBSqlBuilder) Select(fields ...string) SqlBuilder {
	d.fields = fields
	return d
}

// From 实现 From 方法
func (d *DuckDBSqlBuilder) From(formId string) SqlBuilder {
	d.formId = formId
	return d
}

func (d *DuckDBSqlBuilder) GetWhere() []Where {
	return d.where
}

// Where 实现 Where 方法
func (d *DuckDBSqlBuilder) Where(sql interface{}, args ...interface{}) SqlBuilder {
	switch sql.(type) {
	case string:
		d.where = append(d.where, Where{
			Sql:       sql.(string),
			WhereType: "AND",
			Args:      args,
		})
	case SqlBuilder:
		d.where = append(d.where, Where{
			Sql:       "",
			WhereType: "AND",
			Child:     sql.(SqlBuilder).GetWhere(),
		})
	}

	return d
}

// WhereOr 实现 WhereOr 方法
func (d *DuckDBSqlBuilder) WhereOr(sql interface{}, args ...interface{}) SqlBuilder {
	switch sql.(type) {
	case string:
		d.where = append(d.where, Where{
			Sql:       sql.(string),
			WhereType: "OR",
			Args:      args,
		})
	case SqlBuilder:
		d.where = append(d.where, Where{
			Sql:       "",
			WhereType: "OR",
			Child:     sql.(SqlBuilder).GetWhere(),
		})
	}

	return d
}

// Limit 方法
// 如果是2个以上参数，则第一个参数为页码，第二个参数为每页数量
// 如果是1个参数，则参数为数量
func (d *DuckDBSqlBuilder) Limit(limitParam ...int) SqlBuilder {
	if len(limitParam) == 1 {
		d.limitSize = limitParam[0]
	} else if len(limitParam) >= 2 {
		d.limitPage = limitParam[0]
		d.limitSize = limitParam[1]
	}
	return d
}

// GroupBy 实现 GroupBy 方法
func (d *DuckDBSqlBuilder) GroupBy(groupBy string) SqlBuilder {
	d.groupBy = groupBy
	return d
}

// OrderBy 实现 OrderBy 方法
func (d *DuckDBSqlBuilder) OrderBy(orderBy string) SqlBuilder {
	d.orderBy = orderBy
	return d
}

func (d *DuckDBSqlBuilder) buildWhere(wheres []Where) string {
	sql := ""
	for k, where := range wheres {
		_sql := ""
		if where.Sql != "" {
			_sql = where.Sql
		} else if where.Child != nil {
			_sql = d.buildWhere(where.Child)
		}

		if k == 0 {
			sql += " ( " + _sql + " ) "
		} else {
			sql += " " + where.WhereType + " ( " + _sql + " ) "
		}
		if len(where.Args) > 0 {
			d.whereArgs = append(d.whereArgs, where.Args...)
		}
	}
	return sql
}

func (d *DuckDBSqlBuilder) Build() (string, []interface{}) {
	sql := ""
	if len(d.fields) > 0 {
		sql += "SELECT " + strings.Join(d.fields, ",")
	}
	if d.formId != "" {
		sql += " FROM " + d.formId
	}
	if len(d.where) > 0 {
		sql += fmt.Sprintf(" WHERE %s", d.buildWhere(d.where))
	}
	if d.groupBy != "" {
		sql += " GROUP BY " + d.groupBy
	}
	if d.orderBy != "" {
		sql += " ORDER BY " + d.orderBy
	}
	if d.limitSize > 0 {
		sql += fmt.Sprintf(" LIMIT %d", d.limitSize)
	}
	if d.limitPage > 0 {
		sql += fmt.Sprintf(" OFFSET %d", d.limitSize*(d.limitPage-1))
	}

	return sql, d.whereArgs
}
