package ndbclient

import (
	"context"
	"fmt"
)

type DataSqlModel struct {
	groupId string
	limit   int
	offset  int
	sql     string
	args    []interface{}
	detail  map[string]interface{}
	update  map[string]interface{}
	del     map[string]interface{}
	formId  string
	ctx     context.Context
}

func SqlModel(groupId string, formId string) *DataSqlModel {
	d := &DataSqlModel{
		groupId: groupId,
		formId:  formId,
	}
	return d
}

func (d *DataSqlModel) SqlBuilder(sqlBuilder SqlBuilder) *DataSqlModel {
	d.sql, d.args = sqlBuilder.Build()
	return d
}

func (d *DataSqlModel) Sql(sql string, args ...interface{}) *DataSqlModel {
	d.sql, d.args = sql, args
	return d
}

// Limit
// 如果是2个以上参数，则第一个参数为页码，第二个参数为每页数量
// 如果是1个参数，则参数为数量
// example:1,10
func (d *DataSqlModel) Limit(limitParam ...int) *DataSqlModel {
	if len(limitParam) == 1 {
		d.limit = limitParam[0]
	} else if len(limitParam) >= 2 {
		d.offset = (limitParam[0] - 1) * limitParam[1]
		d.limit = limitParam[1]
	}
	return d
}

func (d *DataSqlModel) Ctx(ctx context.Context) *DataSqlModel {
	d.ctx = ctx
	return d
}

func (d *DataSqlModel) Find() Response {
	if d.offset != 0 {
		d.sql += fmt.Sprintf(" OFFSET %d", d.offset)
	}
	db := Sql(d.groupId).SetSql(d.sql, d.args...)
	if d.limit != 0 {
		db.SetLimit(d.limit)
	}
	return db.Do(context.Background())
}

// Update 更新内容
//
//	example:  updateData ：=map[string]interface{}{
//				"name": "john2",
//			}
func (d *DataSqlModel) Update(updateData map[string]interface{}) Response {
	db := Sql(d.groupId).SetSql(d.sql, d.args...)
	if d.limit != 0 {
		db.SetLimit(d.limit)
	}
	return db.SetUpdate(map[string]interface{}{
		"form_id": d.formId,
		"set":     updateData,
	}).Do(context.Background())
}

// Delete 删除，默认是软删除
func (d *DataSqlModel) Delete(trueDel ...bool) Response {
	db := Sql(d.groupId).SetSql(d.sql, d.args...)
	if d.limit != 0 {
		db.SetLimit(d.limit)
	}
	softDel := 1
	if len(trueDel) > 0 && trueDel[0] {
		softDel = 0
	}
	return db.SetDel(map[string]interface{}{
		"form_id": d.formId,
		"softdel": softDel,
	}).Do(context.Background())
}
